self: { lib, flake-parts-lib, ... }:
let
  inherit (flake-parts-lib) mkPerSystemOption;
  inherit (lib) types mkOption;
in
{
  options = {
    perSystem = mkPerSystemOption ({ config, self', inputs', pkgs, system, ... }: {
      options = {
        document.variables = mkOption {
          type = types.attrsOf types.str;
          default = { };
        };
        document.source = mkOption {
          type = types.path;
        };
        document.mainFile = mkOption {
          type = types.str;
          default = "main";
        };
        document.texlive = mkOption {
          type = types.package;
          default = pkgs.texlive.combine ({
            inherit (pkgs.texlive) scheme-basic latexmk pgf fontspec
              latexindent lipsum hyperref anysize natbib fancyhdr
              multirow xcolor soul parskip etoolbox mathtools amsmath
              amstex dirtytalk float enumitem svg polyglossia babel-hungarian babel
              latex-uni8 graphics geometry biber biblatex tocloft titlesec adjustbox
              bookmark url csquotes listings listings-ext sourcecodepro silence
              biblatex-ieee ly1 metafont transparent catchfile microtype
              l3kernel l3packages texcount moreverb pdfpages pdflscape
              tabularray ninecolors;
          });
        };
        document.font = mkOption {
          type = types.package;
          default = pkgs.recursive;
        };
      };

      config =
        let
          texvars = toString (lib.mapAttrsToList (k: v: ''\def\${k}{${v}}'') config.document.variables);
          mainFile = config.document.mainFile;
          buildCommand = lib.getExe (pkgs.writeShellScriptBin "build-document" ''
            BUILDDATE="$(TZ='Europe/Budapest' LC_TIME='hu_HU.UTF-8' date +'%Y. %b. %d. %T')"
            BUILDCOMMIT="$(git rev-parse --short HEAD || echo '${ if (self ? shortRev) then self.shortRev else "<ismeretlen>"}')"
            VARS="\\def\\builddate{$BUILDDATE} \\def\\buildcommit{$BUILDCOMMIT}"
            latexmk -interaction=nonstopmode -pdf -lualatex \
              -outdir="./build" -synctex=1 --shell-escape \
              -pretex="\relax${texvars} $VARS" \
              -usepretex "${mainFile}.tex"
          '');

          locales = pkgs.glibcLocales.override {
            locales = [
              "en_US.UTF-8/UTF8"
              "hu_HU.UTF-8/UTF8"
            ];
          };
        in
        {
          packages.document = pkgs.stdenvNoCC.mkDerivation {
            name = "latex-document";
            src = config.document.source;
            buildInputs = [ pkgs.coreutils pkgs.inkscape config.document.texlive config.document.font locales ];
            phases = [ "unpackPhase" "buildPhase" "installPhase" ];
            buildPhase = ''
              set -o errexit
              mkdir -p ./build

              TEMP="$(mktemp -d)"
              cleanup () {
                if [ -d "$TEMP" ]; then
                  rm -rf "$TEMP"
                fi
              }
              trap cleanup EXIT

              env TEXMFHOME="$TEMP/.cache" \
                TEXMFVAR="$TEMP/.cache/texf-var" \
                OSFONTDIR=${config.document.font}/share/fonts \
                ${buildCommand}
            '';
            installPhase = ''
              mkdir -p $out
              mv -v ./build/*.pdf $out/.
            '';
          };

          devShells.document =
            let
              doc-watcher = pkgs.writeShellApplication {
                name = "doc-watcher";
                runtimeInputs = with pkgs; [ watchexec coreutils inkscape ] ++ [ config.document.texlive ];
                text = ''
                  export OSFONTDIR="${config.document.font}/share/fonts"
                  watchexec -r --print-events -- \
                    ${buildCommand}
                '';
              };
            in
            pkgs.mkShell {
              name = "latex-document-shell";
              buildInputs = [ config.document.texlive pkgs.texlab pkgs.inkscape doc-watcher ];
            };
        };
    });
  };
}
